/**
 * Gestionnaire des suppressions avec confirmation
 */
class DeletionManager {
    constructor() {
        this.init();
    }

    init() {
        // Attacher les événements aux boutons de suppression
        document.addEventListener('DOMContentLoaded', () => {
            this.attachDeleteEvents();
        });
    }

    attachDeleteEvents() {
        // Gestionnaire pour les formulaires de suppression avec classe 'delete-form'
        const deleteForms = document.querySelectorAll('.delete-form');
        deleteForms.forEach(form => {
            form.addEventListener('submit', (e) => {
                e.preventDefault();
                this.showDeleteConfirmation(form);
            });
        });

        // Gestionnaire pour les boutons avec attribut data-delete
        const deleteButtons = document.querySelectorAll('[data-delete]');
        deleteButtons.forEach(button => {
            button.addEventListener('click', (e) => {
                e.preventDefault();
                const form = button.closest('form') || document.querySelector(button.dataset.delete);
                if (form) {
                    this.showDeleteConfirmation(form);
                }
            });
        });

        // Gestionnaire pour les liens de suppression avec classe 'delete-link'
        const deleteLinks = document.querySelectorAll('.delete-link');
        deleteLinks.forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                this.showDeleteConfirmation(null, link.href, link.dataset.method || 'DELETE');
            });
        });
    }

    showDeleteConfirmation(form, url = null, method = 'DELETE') {
        const itemName = form?.dataset.itemName || 'cet élément';
        const itemType = form?.dataset.itemType || 'élément';
        
        const modal = this.createConfirmationModal(itemName, itemType);
        document.body.appendChild(modal);

        // Gérer les actions des boutons
        const confirmBtn = modal.querySelector('#confirmDelete');
        const cancelBtn = modal.querySelector('#cancelDelete');

        confirmBtn.addEventListener('click', () => {
            if (form) {
                form.submit();
            } else if (url) {
                this.submitDeleteRequest(url, method);
            }
            this.closeModal(modal);
        });

        cancelBtn.addEventListener('click', () => {
            this.closeModal(modal);
        });

        // Fermer avec Escape
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape') {
                this.closeModal(modal);
            }
        });

        // Afficher le modal
        modal.style.display = 'flex';
        setTimeout(() => {
            modal.classList.add('show');
        }, 10);
    }

    createConfirmationModal(itemName, itemType) {
        const modal = document.createElement('div');
        modal.className = 'fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 opacity-0 transition-opacity duration-300';
        modal.innerHTML = `
            <div class="bg-white rounded-lg p-6 max-w-md w-full mx-4 transform scale-95 transition-transform duration-300">
                <div class="flex items-center mb-4">
                    <div class="flex-shrink-0">
                        <svg class="w-6 h-6 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.996-.833-2.764 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z"></path>
                        </svg>
                    </div>
                    <div class="ml-3">
                        <h3 class="text-lg font-medium text-gray-900">Confirmer la suppression</h3>
                    </div>
                </div>
                
                <div class="mb-6">
                    <p class="text-sm text-gray-500">
                        Êtes-vous sûr de vouloir supprimer <strong>${itemName}</strong> ?
                        <br><br>
                        Cette action est <strong>irréversible</strong> et supprimera définitivement ce ${itemType} 
                        ainsi que tous les fichiers associés.
                    </p>
                </div>
                
                <div class="flex justify-end space-x-3">
                    <button id="cancelDelete" type="button" class="px-4 py-2 text-sm font-medium text-gray-700 bg-gray-100 border border-gray-300 rounded-md hover:bg-gray-200 focus:outline-none focus:ring-2 focus:ring-gray-500">
                        Annuler
                    </button>
                    <button id="confirmDelete" type="button" class="px-4 py-2 text-sm font-medium text-white bg-red-600 border border-transparent rounded-md hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-red-500">
                        Supprimer définitivement
                    </button>
                </div>
            </div>
        `;
        return modal;
    }

    submitDeleteRequest(url, method) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.action = url;

        // Token CSRF
        const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content');
        if (csrfToken) {
            const csrfInput = document.createElement('input');
            csrfInput.type = 'hidden';
            csrfInput.name = '_token';
            csrfInput.value = csrfToken;
            form.appendChild(csrfInput);
        }

        // Méthode HTTP
        if (method !== 'POST') {
            const methodInput = document.createElement('input');
            methodInput.type = 'hidden';
            methodInput.name = '_method';
            methodInput.value = method;
            form.appendChild(methodInput);
        }

        document.body.appendChild(form);
        form.submit();
    }

    closeModal(modal) {
        modal.classList.remove('show');
        setTimeout(() => {
            if (modal.parentNode) {
                modal.parentNode.removeChild(modal);
            }
        }, 300);
    }

    // Méthode utilitaire pour ajouter dynamiquement un gestionnaire de suppression
    static addDeleteHandler(element, itemName, itemType = 'élément') {
        element.dataset.itemName = itemName;
        element.dataset.itemType = itemType;
        
        element.addEventListener('click', (e) => {
            e.preventDefault();
            const manager = new DeletionManager();
            const form = element.closest('form');
            if (form) {
                manager.showDeleteConfirmation(form);
            }
        });
    }
}

// Initialiser le gestionnaire
const deletionManager = new DeletionManager();

// Exporter pour utilisation globale
window.DeletionManager = DeletionManager;