<?php

use Illuminate\Support\Facades\Config;

if (!function_exists('asu_config')) {
    /**
     * Récupère une valeur de configuration ASU
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    function asu_config($key = null, $default = null)
    {
        if (is_null($key)) {
            return Config::get('asu');
        }

        return Config::get("asu.{$key}", $default);
    }
}

if (!function_exists('asu_social')) {
    /**
     * Récupère les liens des réseaux sociaux ASU
     *
     * @param string $platform
     * @return string|array
     */
    function asu_social($platform = null)
    {
        $social = asu_config('social', []);

        if (is_null($platform)) {
            return $social;
        }

        return $social[$platform] ?? '#';
    }
}

if (!function_exists('asu_contact')) {
    /**
     * Récupère les informations de contact ASU
     *
     * @param string $type
     * @return string|array
     */
    function asu_contact($type = null)
    {
        $contact = asu_config('contact', []);

        if (is_null($type)) {
            return $contact;
        }

        return $contact[$type] ?? '';
    }
}

if (!function_exists('asu_stats')) {
    /**
     * Récupère les statistiques ASU
     *
     * @param string $stat
     * @return int|array
     */
    function asu_stats($stat = null)
    {
        $stats = asu_config('stats', []);

        if (is_null($stat)) {
            return $stats;
        }

        return $stats[$stat] ?? 0;
    }
}

if (!function_exists('format_large_number')) {
    /**
     * Formate un grand nombre pour l'affichage
     *
     * @param int $number
     * @return string
     */
    function format_large_number($number)
    {
        if ($number >= 1000000) {
            return round($number / 1000000, 1) . 'M';
        } elseif ($number >= 1000) {
            return round($number / 1000, 1) . 'K';
        }

        return number_format($number);
    }
}

if (!function_exists('generate_youtube_embed')) {
    /**
     * Génère un lien d'embed YouTube à partir d'une URL
     *
     * @param string $url
     * @return string
     */
    function generate_youtube_embed($url)
    {
        $pattern = '/(?:youtube\.com\/(?:[^\/]+\/.+\/|(?:v|e(?:mbed)?)\/|.*[?&]v=)|youtu\.be\/)([^"&?\/\s]{11})/';
        
        if (preg_match($pattern, $url, $matches)) {
            return "https://www.youtube.com/embed/{$matches[1]}";
        }
        
        return $url;
    }
}

if (!function_exists('truncate_text')) {
    /**
     * Tronque un texte à un nombre de mots donné
     *
     * @param string $text
     * @param int $limit
     * @param string $end
     * @return string
     */
    function truncate_text($text, $limit = 20, $end = '...')
    {
        $words = explode(' ', $text);
        
        if (count($words) > $limit) {
            return implode(' ', array_slice($words, 0, $limit)) . $end;
        }
        
        return $text;
    }
}

if (!function_exists('seo_title')) {
    /**
     * Génère un titre SEO optimisé
     *
     * @param string $title
     * @param bool $includeOrg
     * @return string
     */
    function seo_title($title = null, $includeOrg = true)
    {
        $orgName = asu_config('name');
        
        if (is_null($title)) {
            return asu_config('seo.default_title');
        }
        
        if ($includeOrg) {
            return "{$title} | {$orgName}";
        }
        
        return $title;
    }
}

if (!function_exists('age_since_founded')) {
    /**
     * Calcule l'âge de l'association
     *
     * @return int
     */
    function age_since_founded()
    {
        return date('Y') - asu_config('founded_year', 1998);
    }
}

if (!function_exists('formatBytes')) {
    /**
     * Format bytes into human readable format
     */
    function formatBytes($bytes, $precision = 2)
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
}

if (!function_exists('asset_storage')) {
    /**
     * Generate URL for storage files with fallback
     */
    function asset_storage($path, $fallback = null)
    {
        if (!$path) {
            return $fallback ? asset($fallback) : null;
        }

        // Vérifier si le fichier existe
        if (\Storage::disk('public')->exists($path)) {
            return asset('storage/' . $path);
        }

        return $fallback ? asset($fallback) : null;
    }
}

if (!function_exists('image_url')) {
    /**
     * Generate image URL with fallback
     */
    function image_url($path, $fallback = 'images/placeholder.jpg')
    {
        return asset_storage($path, $fallback);
    }
}

if (!function_exists('file_exists_storage')) {
    /**
     * Check if file exists in storage
     */
    function file_exists_storage($path)
    {
        if (!$path) {
            return false;
        }

        return \Storage::disk('public')->exists($path);
    }
}

if (!function_exists('get_file_size_storage')) {
    /**
     * Get file size from storage
     */
    function get_file_size_storage($path)
    {
        if (!$path || !file_exists_storage($path)) {
            return 0;
        }

        try {
            return \Storage::disk('public')->size($path);
        } catch (\Exception $e) {
            return 0;
        }
    }
}

if (!function_exists('is_image_file')) {
    /**
     * Check if file is an image based on extension
     */
    function is_image_file($path)
    {
        if (!$path) {
            return false;
        }

        $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'];
        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        
        return in_array($extension, $imageExtensions);
    }
}

if (!function_exists('get_file_icon')) {
    /**
     * Get file icon based on extension
     */
    function get_file_icon($path)
    {
        if (!$path) {
            return 'fas fa-file';
        }

        $extension = strtolower(pathinfo($path, PATHINFO_EXTENSION));
        
        $icons = [
            'pdf' => 'fas fa-file-pdf text-red-500',
            'doc' => 'fas fa-file-word text-blue-500',
            'docx' => 'fas fa-file-word text-blue-500',
            'xls' => 'fas fa-file-excel text-green-500',
            'xlsx' => 'fas fa-file-excel text-green-500',
            'ppt' => 'fas fa-file-powerpoint text-orange-500',
            'pptx' => 'fas fa-file-powerpoint text-orange-500',
            'zip' => 'fas fa-file-archive text-yellow-500',
            'rar' => 'fas fa-file-archive text-yellow-500',
            'jpg' => 'fas fa-file-image text-purple-500',
            'jpeg' => 'fas fa-file-image text-purple-500',
            'png' => 'fas fa-file-image text-purple-500',
            'gif' => 'fas fa-file-image text-purple-500',
            'webp' => 'fas fa-file-image text-purple-500',
            'mp4' => 'fas fa-file-video text-indigo-500',
            'avi' => 'fas fa-file-video text-indigo-500',
            'mov' => 'fas fa-file-video text-indigo-500',
            'mp3' => 'fas fa-file-audio text-pink-500',
            'wav' => 'fas fa-file-audio text-pink-500',
            'txt' => 'fas fa-file-alt text-gray-500',
        ];
        
        return $icons[$extension] ?? 'fas fa-file text-gray-500';
    }
}

if (!function_exists('setting')) {
    /**
     * Get setting value by key
     */
    function setting($key, $default = null)
    {
        static $settings = null;
        
        if ($settings === null) {
            try {
                $settings = \App\Models\Setting::where('is_active', true)
                    ->pluck('value', 'key')
                    ->toArray();
            } catch (\Exception $e) {
                $settings = [];
            }
        }
        
        return $settings[$key] ?? $default;
    }
}

if (!function_exists('setting_file_url')) {
    /**
     * Get setting file URL
     */
    function setting_file_url($key, $fallback = null)
    {
        $value = setting($key);
        return asset_storage($value, $fallback);
    }
}

if (!function_exists('truncate_text')) {
    /**
     * Truncate text to specified length
     */
    function truncate_text($text, $length = 100, $suffix = '...')
    {
        if (strlen($text) <= $length) {
            return $text;
        }
        
        return substr($text, 0, $length) . $suffix;
    }
}