<?php

namespace App\Traits;

use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Gd\Driver;

trait ManagesFiles
{
    /**
     * Upload un fichier et retourne le chemin
     */
    protected function uploadFile(UploadedFile $file, string $directory = 'uploads', string $disk = 'public'): string
    {
        $filename = $this->generateUniqueFilename($file->getClientOriginalName());
        return $file->storeAs($directory, $filename, $disk);
    }

    /**
     * Upload une image avec redimensionnement optionnel
     */
    protected function uploadImage(UploadedFile $file, string $directory = 'images', int $maxWidth = 1920, int $maxHeight = 1080, string $disk = 'public'): string
    {
        $filename = $this->generateUniqueFilename($file->getClientOriginalName());
        $path = $file->storeAs($directory, $filename, $disk);

        // Redimensionner l'image si nécessaire
        try {
            $fullPath = Storage::disk($disk)->path($path);
            $manager = new ImageManager(new Driver());
            $image = $manager->read($fullPath);
            
            // Redimensionner seulement si l'image est plus grande que les dimensions max
            if ($image->width() > $maxWidth || $image->height() > $maxHeight) {
                $image->scale($maxWidth, $maxHeight);
                $image->save($fullPath);
            }
        } catch (\Exception $e) {
            \Log::warning("Impossible de redimensionner l'image {$path}: " . $e->getMessage());
        }
        
        return $path;
    }

    /**
     * Upload une image avec création de thumbnail
     */
    protected function uploadImageWithThumbnail(UploadedFile $file, string $directory = 'images', string $thumbnailDirectory = 'thumbnails', string $disk = 'public'): array
    {
        $filename = $this->generateUniqueFilename($file->getClientOriginalName());
        $imagePath = $file->storeAs($directory, $filename, $disk);
        
        try {
            // Créer le thumbnail
            $thumbnailFilename = 'thumb_' . $filename;
            $thumbnailPath = $thumbnailDirectory . '/' . $thumbnailFilename;
            
            $fullPath = Storage::disk($disk)->path($imagePath);
            $thumbnailFullPath = Storage::disk($disk)->path($thumbnailPath);
            
            // Créer le dossier thumbnail s'il n'existe pas
            Storage::disk($disk)->makeDirectory($thumbnailDirectory);
            
            $manager = new ImageManager(new Driver());
            $image = $manager->read($fullPath);
            
            // Créer thumbnail 300x300
            $image->cover(300, 300);
            $image->save($thumbnailFullPath);
            
            return [
                'image' => $imagePath,
                'thumbnail' => $thumbnailPath
            ];
        } catch (\Exception $e) {
            \Log::warning("Impossible de créer le thumbnail pour {$imagePath}: " . $e->getMessage());
            return [
                'image' => $imagePath,
                'thumbnail' => null
            ];
        }
    }

    /**
     * Supprime un fichier de manière sécurisée
     */
    protected function deleteFile(?string $filePath, string $disk = 'public'): bool
    {
        if (!$filePath) {
            return true;
        }

        try {
            if (Storage::disk($disk)->exists($filePath)) {
                return Storage::disk($disk)->delete($filePath);
            }
            return true;
        } catch (\Exception $e) {
            \Log::error("Erreur lors de la suppression du fichier {$filePath}: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Supprime plusieurs fichiers
     */
    protected function deleteFiles(array $filePaths, string $disk = 'public'): bool
    {
        $success = true;
        foreach ($filePaths as $filePath) {
            if (!$this->deleteFile($filePath, $disk)) {
                $success = false;
            }
        }
        return $success;
    }

    /**
     * Valide un fichier uploadé
     */
    protected function validateFileUpload(UploadedFile $file, array $allowedTypes = [], int $maxSize = 10240): bool
    {
        // Vérifier la taille (en KB)
        if ($file->getSize() > ($maxSize * 1024)) {
            return false;
        }

        // Vérifier le type MIME
        if (!empty($allowedTypes) && !in_array($file->getMimeType(), $allowedTypes)) {
            return false;
        }

        return true;
    }

    /**
     * Obtient l'URL publique d'un fichier
     */
    protected function getFileUrl(?string $filePath, string $disk = 'public'): ?string
    {
        if (!$filePath) {
            return null;
        }

        try {
            if (Storage::disk($disk)->exists($filePath)) {
                return Storage::disk($disk)->url($filePath);
            }
            return null;
        } catch (\Exception $e) {
            \Log::warning("Impossible d'obtenir l'URL pour {$filePath}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Obtient la taille d'un fichier en format lisible
     */
    protected function getFileSize(string $filePath, string $disk = 'public'): string
    {
        try {
            if (Storage::disk($disk)->exists($filePath)) {
                $bytes = Storage::disk($disk)->size($filePath);
                return $this->formatBytes($bytes);
            }
            return 'Fichier non trouvé';
        } catch (\Exception $e) {
            return 'Erreur';
        }
    }

    /**
     * Formate la taille en bytes en format lisible
     */
    protected function formatBytes(int $bytes, int $precision = 2): string
    {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }

    /**
     * Crée un nom de fichier unique
     */
    protected function generateUniqueFilename(string $originalName): string
    {
        $extension = pathinfo($originalName, PATHINFO_EXTENSION);
        $name = pathinfo($originalName, PATHINFO_FILENAME);
        
        $slug = Str::slug($name);
        $timestamp = time();
        $random = Str::random(8);
        
        return "{$slug}_{$timestamp}_{$random}.{$extension}";
    }

    /**
     * Nettoie le nom de fichier
     */
    protected function sanitizeFilename(string $filename): string
    {
        $filename = preg_replace('/[^a-zA-Z0-9._-]/', '_', $filename);
        $filename = preg_replace('/_+/', '_', $filename);
        return trim($filename, '_');
    }

    /**
     * Vérifie si un fichier est une image
     */
    protected function isImage(UploadedFile $file): bool
    {
        $imageTypes = [
            'image/jpeg',
            'image/jpg', 
            'image/png',
            'image/gif',
            'image/webp',
            'image/svg+xml'
        ];
        
        return in_array($file->getMimeType(), $imageTypes);
    }

    /**
     * Obtient les dimensions d'une image
     */
    protected function getImageDimensions(string $filePath, string $disk = 'public'): ?array
    {
        try {
            if (!Storage::disk($disk)->exists($filePath)) {
                return null;
            }

            $fullPath = Storage::disk($disk)->path($filePath);
            $imageInfo = getimagesize($fullPath);
            
            if ($imageInfo) {
                return [
                    'width' => $imageInfo[0],
                    'height' => $imageInfo[1],
                    'type' => $imageInfo['mime']
                ];
            }
            
            return null;
        } catch (\Exception $e) {
            \Log::warning("Impossible d'obtenir les dimensions de {$filePath}: " . $e->getMessage());
            return null;
        }
    }
}