<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;

class Project extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'slug',
        'description',
        'content',
        'featured_image',
        'status',
        'budget',
        'currency',
        'start_date',
        'end_date',
        'progress_percentage',
        'objectives',
        'beneficiaries',
        'location',
        'partners',
        'is_featured',
        'order',
    ];

    protected $casts = [
        'objectives' => 'array',
        'beneficiaries' => 'array',
        'partners' => 'array',
        'is_featured' => 'boolean',
        'budget' => 'decimal:2',
        'progress_percentage' => 'integer',
        'order' => 'integer',
        'start_date' => 'date',
        'end_date' => 'date',
    ];

    // Scopes
    public function scopeActive(Builder $query): Builder
    {
        return $query->where('status', 'in_progress');
    }

    public function scopeCompleted(Builder $query): Builder
    {
        return $query->where('status', 'completed');
    }

    public function scopeFeatured(Builder $query): Builder
    {
        return $query->where('is_featured', true);
    }

    public function scopeOrdered(Builder $query): Builder
    {
        return $query->orderBy('order');
    }

    // Mutators
    public function setTitleAttribute($value)
    {
        $this->attributes['title'] = $value;
        $this->attributes['slug'] = Str::slug($value);
    }

    // Accessors
    public function getStatusLabelAttribute()
    {
        $labels = [
            'planned' => 'Planifié',
            'in_progress' => 'En cours',
            'completed' => 'Terminé',
            'on_hold' => 'En pause',
        ];

        return $labels[$this->status] ?? $this->status;
    }

    public function getStatusColorAttribute()
    {
        $colors = [
            'planned' => 'blue',
            'in_progress' => 'green',
            'completed' => 'orange',
            'on_hold' => 'gray',
        ];

        return $colors[$this->status] ?? 'gray';
    }

    public function getBudgetFormattedAttribute()
    {
        if (!$this->budget) return null;
        
        return number_format($this->budget, 0, ',', ' ') . ' ' . $this->currency;
    }

    public function getDurationAttribute()
    {
        if (!$this->start_date || !$this->end_date) return null;
        
        return $this->start_date->diffInMonths($this->end_date);
    }

    public function getIsActiveAttribute()
    {
        return $this->status === 'in_progress';
    }

    public function getRouteKeyName()
    {
        return 'slug';
    }
}
