<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;

class Gallery extends Model
{
    use HasFactory;

    protected $fillable = [
        'title',
        'description',
        'image_path',
        'thumbnail_path',
        'type',
        'video_url',
        'alt_text',
        'category',
        'event_date',
        'location',
        'tags',
        'is_featured',
        'is_published',
        'order',
        'views',
        'meta',
    ];

    protected $casts = [
        'tags' => 'array',
        'meta' => 'array',
        'is_featured' => 'boolean',
        'is_published' => 'boolean',
        'order' => 'integer',
        'views' => 'integer',
        'event_date' => 'date',
    ];

    // Scopes
    public function scopePublished(Builder $query): Builder
    {
        return $query->where('is_published', true);
    }

    public function scopeFeatured(Builder $query): Builder
    {
        return $query->where('is_featured', true);
    }

    public function scopeByCategory(Builder $query, string $category): Builder
    {
        return $query->where('category', $category);
    }

    public function scopeByType(Builder $query, string $type): Builder
    {
        return $query->where('type', $type);
    }

    public function scopeOrdered(Builder $query): Builder
    {
        return $query->orderBy('order');
    }

    // Accessors
    public function getImageUrlAttribute()
    {
        return $this->image_path 
            ? asset('storage/' . $this->image_path)
            : null;
    }

    public function getThumbnailUrlAttribute()
    {
        return $this->thumbnail_path 
            ? asset('storage/' . $this->thumbnail_path)
            : $this->image_url;
    }

    public function getCategoryLabelAttribute()
    {
        $labels = [
            'events' => 'Événements',
            'training' => 'Formations',
            'community' => 'Communauté',
            'projects' => 'Projets',
            'partnerships' => 'Partenariats',
            'general' => 'Général',
        ];

        return $labels[$this->category] ?? $this->category;
    }

    public function getIsVideoAttribute()
    {
        return $this->type === 'video';
    }

    // Methods
    public function incrementViews()
    {
        $this->increment('views');
    }
}
