<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class CheckDeletePermission
{
    /**
     * Handle an incoming request.
     */
    public function handle(Request $request, Closure $next, string $model = null): Response
    {
        // Vérifier si l'utilisateur est connecté
        if (!auth()->check()) {
            return redirect()->route('login')
                ->with('error', 'Vous devez être connecté pour effectuer cette action.');
        }

        // Vérifier si l'utilisateur a les permissions admin
        $user = auth()->user();
        if (!$user->is_admin && !$user->hasRole('admin')) {
            return redirect()->back()
                ->with('error', 'Vous n\'avez pas les permissions nécessaires pour supprimer cet élément.');
        }

        // Vérifications spécifiques par modèle
        if ($model) {
            $result = $this->checkModelSpecificRules($request, $model);
            if ($result !== true) {
                return $result;
            }
        }

        return $next($request);
    }

    /**
     * Vérifications spécifiques par modèle
     */
    private function checkModelSpecificRules(Request $request, string $model): mixed
    {
        switch ($model) {
            case 'page':
                return $this->checkPageDeletion($request);
            
            case 'user':
                return $this->checkUserDeletion($request);
            
            case 'setting':
                return $this->checkSettingDeletion($request);
            
            default:
                return true;
        }
    }

    /**
     * Vérifications pour la suppression de pages
     */
    private function checkPageDeletion(Request $request): mixed
    {
        $pageId = $request->route('page')?->id ?? $request->route()->parameter('page');
        
        if ($pageId) {
            $page = \App\Models\Page::find($pageId);
            
            if ($page && $page->children()->count() > 0) {
                return redirect()->back()
                    ->with('error', 'Impossible de supprimer une page qui a des sous-pages. Supprimez d\'abord les sous-pages.');
            }
            
            // Empêcher la suppression de pages système importantes
            $protectedSlugs = ['accueil', 'contact', 'a-propos', 'mentions-legales'];
            if ($page && in_array($page->slug, $protectedSlugs)) {
                return redirect()->back()
                    ->with('error', 'Cette page est protégée et ne peut pas être supprimée.');
            }
        }
        
        return true;
    }

    /**
     * Vérifications pour la suppression d'utilisateurs
     */
    private function checkUserDeletion(Request $request): mixed
    {
        $userId = $request->route('user')?->id ?? $request->route()->parameter('user');
        $currentUser = auth()->user();
        
        // Empêcher l'auto-suppression
        if ($userId == $currentUser->id) {
            return redirect()->back()
                ->with('error', 'Vous ne pouvez pas supprimer votre propre compte.');
        }
        
        // Empêcher la suppression du dernier admin
        if ($userId) {
            $user = \App\Models\User::find($userId);
            if ($user && $user->is_admin) {
                $adminCount = \App\Models\User::where('is_admin', true)->count();
                if ($adminCount <= 1) {
                    return redirect()->back()
                        ->with('error', 'Impossible de supprimer le dernier administrateur.');
                }
            }
        }
        
        return true;
    }

    /**
     * Vérifications pour la suppression de paramètres
     */
    private function checkSettingDeletion(Request $request): mixed
    {
        $settingId = $request->route('setting')?->id ?? $request->route()->parameter('setting');
        
        if ($settingId) {
            $setting = \App\Models\Setting::find($settingId);
            
            // Protéger les paramètres système critiques
            $protectedKeys = [
                'site_name',
                'site_description', 
                'admin_email',
                'app_environment',
                'app_debug'
            ];
            
            if ($setting && in_array($setting->key, $protectedKeys)) {
                return redirect()->back()
                    ->with('error', 'Ce paramètre système ne peut pas être supprimé.');
            }
        }
        
        return true;
    }
}