<?php

namespace App\Http\Controllers;

use App\Models\Article;
use App\Models\Project;
use App\Models\Testimonial;
use App\Models\Gallery;
use App\Models\Partner;
use App\Models\Faq;
use App\Models\Document;
use App\Models\Setting;
use Illuminate\Http\Request;

class HomeController extends Controller
{
    /**
     * Homepage with dynamic content
     */
    public function index()
    {
        try {
            // Récupérer les données pour la homepage avec gestion d'erreurs
            $featuredArticles = collect();
            $latestArticles = collect();
            $activeProjects = collect();
            $featuredTestimonials = collect();
            $galleryImages = collect();
            $activePartners = collect();
            $featuredFaqs = collect();
            $featuredDocuments = collect();
            $allProjects = collect();
            $allArticles = collect();
            $allPartners = collect();
            
            try {
                $latestArticles = Article::published()->latest()->take(6)->get();
                $allArticles = Article::published()->get(); // Pour les statistiques
            } catch (\Exception $e) {
                \Log::warning('Erreur lors de la récupération des articles: ' . $e->getMessage());
            }
            
            try {
                $activeProjects = Project::latest()->take(3)->get();
                $allProjects = Project::all(); // Pour les statistiques
            } catch (\Exception $e) {
                \Log::warning('Erreur lors de la récupération des projets: ' . $e->getMessage());
            }
            
            try {
                $featuredTestimonials = Testimonial::take(3)->get();
            } catch (\Exception $e) {
                \Log::warning('Erreur lors de la récupération des témoignages: ' . $e->getMessage());
            }
            
            try {
                $activePartners = Partner::take(8)->get();
                $allPartners = Partner::all(); // Pour les statistiques
            } catch (\Exception $e) {
                \Log::warning('Erreur lors de la récupération des partenaires: ' . $e->getMessage());
            }
            
            try {
                $featuredFaqs = Faq::take(5)->get();
            } catch (\Exception $e) {
                \Log::warning('Erreur lors de la récupération des FAQs: ' . $e->getMessage());
            }
            
            try {
                $featuredDocuments = Document::public()->take(6)->get();
            } catch (\Exception $e) {
                \Log::warning('Erreur lors de la récupération des documents: ' . $e->getMessage());
            }
            
            // Statistiques dynamiques avec valeurs par défaut
            $stats = [
                'years_active' => now()->year - 1998,
                'articles_count' => $latestArticles->count(),
                'projects_count' => $activeProjects->count(),
                'partners_count' => 8,
            ];
            
            // Paramètres du site avec valeurs par défaut
            $settings = [
                'site_name' => 'Association Serment Universel',
                'site_description' => 'Depuis 27 ans, nous œuvrons pour la prévention du VIH/SIDA et l\'accompagnement psychosocial en République du Congo',
            ];
            
            try {
                $dbSettings = Setting::getPublicSettings();
                $settings = array_merge($settings, $dbSettings->toArray());
            } catch (\Exception $e) {
                \Log::warning('Erreur lors de la récupération des paramètres: ' . $e->getMessage());
            }

            return view('welcome', compact(
                'featuredArticles',
                'latestArticles', 
                'activeProjects',
                'featuredTestimonials',
                'galleryImages',
                'activePartners',
                'featuredFaqs',
                'featuredDocuments',
                'allProjects',
                'allArticles', 
                'allPartners',
                'settings',
                'stats'
            ));
            
        } catch (\Exception $e) {
            \Log::error('Erreur dans HomeController@index: ' . $e->getMessage());
            
            // Version ultra-simple en cas d'erreur
            return view('welcome', [
                'featuredArticles' => collect(),
                'latestArticles' => collect(),
                'activeProjects' => collect(),
                'featuredTestimonials' => collect(),
                'galleryImages' => collect(),
                'activePartners' => collect(),
                'featuredFaqs' => collect(),
                'featuredDocuments' => collect(),
                'allProjects' => collect(),
                'allArticles' => collect(), 
                'allPartners' => collect(),
                'settings' => [
                    'site_name' => 'Association Serment Universel',
                    'site_description' => 'Depuis 27 ans, nous œuvrons pour la prévention du VIH/SIDA et l\'accompagnement psychosocial en République du Congo',
                ],
                'stats' => [
                    'years_active' => 27,
                    'articles_count' => 0,
                    'projects_count' => 0,
                    'partners_count' => 8,
                ]
            ]);
        }
    }

    /**
     * Articles listing page
     */
    public function articles(Request $request)
    {
        $query = Article::published();

        // Filtres
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('content', 'like', '%' . $request->search . '%');
            });
        }

        $articles = $query->latest()->paginate(9);
        $categories = Article::published()->select('category')->distinct()->pluck('category');
        $featuredArticles = Article::published()->featured()->latest()->take(3)->get();

        return view('articles', compact('articles', 'categories', 'featuredArticles'));
    }

    /**
     * Single article page
     */
    public function article($slug)
    {
        $article = Article::published()->where('slug', $slug)->firstOrFail();
        $relatedArticles = Article::published()
            ->where('category', $article->category)
            ->where('id', '!=', $article->id)
            ->latest()
            ->take(3)
            ->get();

        return view('article', compact('article', 'relatedArticles'));
    }

    /**
     * Projects listing page
     */
    public function projects(Request $request)
    {
        $query = Project::query();

        // Filtres
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        $projects = $query->latest()->paginate(9);
        $activeProjects = Project::active()->latest()->take(3)->get();
        $completedProjects = Project::where('status', 'completed')->count();
        $totalBeneficiaries = Project::sum('beneficiaries');

        return view('projects', compact('projects', 'activeProjects', 'completedProjects', 'totalBeneficiaries'));
    }

    /**
     * Single project page
     */
    public function project($slug)
    {
        $project = Project::where('slug', $slug)->firstOrFail();
        $relatedProjects = Project::where('id', '!=', $project->id)
            ->latest()
            ->take(3)
            ->get();

        return view('project', compact('project', 'relatedProjects'));
    }

    /**
     * About page
     */
    public function about()
    {
        $testimonials = Testimonial::latest()->take(6)->get();
        $stats = [
            'years_active' => Setting::getValue('years_active', '27'),
            'communities_served' => Setting::getValue('communities_served', '150'),
            'people_helped' => Setting::getValue('people_helped', '50000'),
            'volunteers' => Setting::getValue('volunteers', '250'),
        ];

        return view('about', compact('testimonials', 'stats'));
    }

    /**
     * Contact page
     */
    public function contact()
    {
        $settings = Setting::getPublicSettings();
        return view('contact', compact('settings'));
    }

    /**
     * Gallery page
     */
    public function gallery()
    {
        $images = Gallery::byType('image')->ordered()->paginate(12);
        $videos = Gallery::byType('video')->ordered()->take(6)->get();

        return view('gallery', compact('images', 'videos'));
    }

    /**
     * Documents page
     */
    public function documents()
    {
        $documents = Document::public()->latest()->paginate(12);
        $categories = Document::public()->select('category')->distinct()->pluck('category');

        return view('documents', compact('documents', 'categories'));
    }
}