<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Testimonial;
use Illuminate\Http\Request;

class TestimonialController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Testimonial::query();

        // Filtres
        if ($request->filled('featured')) {
            $query->where('featured', $request->featured === 'true');
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('content', 'like', '%' . $request->search . '%')
                  ->orWhere('position', 'like', '%' . $request->search . '%');
            });
        }

        $testimonials = $query->orderBy('created_at', 'desc')->paginate(15);

        return view('admin.testimonials.index', compact('testimonials'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.testimonials.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'title' => 'nullable|string|max:255',
            'content' => 'required|string',
            'rating' => 'nullable|integer|min:1|max:5',
            'is_featured' => 'boolean',
            'avatar' => 'nullable|image|max:1024',
        ]);

        // Gestion de l'avatar
        if ($request->hasFile('avatar')) {
            $validated['avatar'] = $request->file('avatar')->store('testimonials/avatars', 'public');
        }

        Testimonial::create($validated);

        return redirect()->route('admin.testimonials.index')
            ->with('success', 'Témoignage créé avec succès.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Testimonial $testimonial)
    {
        return view('admin.testimonials.show', compact('testimonial'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Testimonial $testimonial)
    {
        return view('admin.testimonials.edit', compact('testimonial'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Testimonial $testimonial)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'title' => 'nullable|string|max:255',
            'content' => 'required|string',
            'rating' => 'nullable|integer|min:1|max:5',
            'is_featured' => 'boolean',
            'avatar' => 'nullable|image|max:1024',
        ]);

        // Gestion de l'avatar
        if ($request->hasFile('avatar')) {
            // Supprimer l'ancien avatar
            if ($testimonial->avatar) {
                \Storage::disk('public')->delete($testimonial->avatar);
            }
            $validated['avatar'] = $request->file('avatar')->store('testimonials/avatars', 'public');
        }

        $testimonial->update($validated);

        return redirect()->route('admin.testimonials.index')
            ->with('success', 'Témoignage mis à jour avec succès.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Testimonial $testimonial)
    {
        try {
            // Supprimer l'avatar
            if ($testimonial->avatar && \Storage::disk('public')->exists($testimonial->avatar)) {
                \Storage::disk('public')->delete($testimonial->avatar);
            }

            // Supprimer le témoignage
            $name = $testimonial->name;
            $testimonial->delete();

            return redirect()->route('admin.testimonials.index')
                ->with('success', "Le témoignage de '{$name}' a été supprimé avec succès.");

        } catch (\Exception $e) {
            \Log::error('Erreur lors de la suppression du témoignage: ' . $e->getMessage());
            
            return redirect()->route('admin.testimonials.index')
                ->with('error', 'Une erreur est survenue lors de la suppression du témoignage.');
        }
    }

    /**
     * Toggle featured status
     */
    public function toggleFeatured(Testimonial $testimonial)
    {
        $testimonial->update(['featured' => !$testimonial->featured]);

        return response()->json([
            'success' => true,
            'featured' => $testimonial->featured
        ]);
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:delete,feature,unfeature',
            'testimonials' => 'required|array',
            'testimonials.*' => 'exists:testimonials,id'
        ]);

        $testimonials = Testimonial::whereIn('id', $request->testimonials);

        switch ($request->action) {
            case 'delete':
                $testimonials->each(function ($testimonial) {
                    if ($testimonial->avatar) {
                        \Storage::disk('public')->delete($testimonial->avatar);
                    }
                });
                $testimonials->delete();
                $message = 'Témoignages supprimés avec succès.';
                break;
            case 'feature':
                $testimonials->update(['featured' => true]);
                $message = 'Témoignages mis en avant avec succès.';
                break;
            case 'unfeature':
                $testimonials->update(['featured' => false]);
                $message = 'Témoignages retirés de la mise en avant avec succès.';
                break;
        }

        return redirect()->route('admin.testimonials.index')
            ->with('success', $message);
    }
}