<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Traits\ManagesFiles;
use Illuminate\Http\Request;

class SettingController extends Controller
{
    use ManagesFiles;
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Setting::query();

        // Filtres
        if ($request->filled('group')) {
            $query->where('group', $request->group);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('key', 'like', '%' . $request->search . '%')
                  ->orWhere('label', 'like', '%' . $request->search . '%')
                  ->orWhere('value', 'like', '%' . $request->search . '%');
            });
        }

        $settings = $query->orderBy('group')->orderBy('order')->paginate(20);
        
        $groups = Setting::select('group')->distinct()->pluck('group');

        return view('admin.settings.index', compact('settings', 'groups'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $groups = ['general', 'contact', 'social', 'stats', 'seo', 'appearance'];
        $types = ['text', 'textarea', 'number', 'url', 'email', 'image', 'boolean', 'json'];
        
        return view('admin.settings.create', compact('groups', 'types'));
    }

        /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'key' => 'required|string|max:255|unique:settings,key',
            'label' => 'required|string|max:255',
            'type' => 'required|string|in:text,textarea,number,email,url,tel,select,checkbox,file,image,color,date,time,json',
            'value' => 'nullable|string',
            'value_textarea' => 'nullable|string',
            'value_json' => 'nullable|string',
            'value_checkbox' => 'nullable|boolean',
            'file' => 'nullable|file|max:10240',
            'description' => 'nullable|string',
            'group' => 'required|string',
            'options' => 'nullable|string',
            'order' => 'nullable|integer',
            'is_required' => 'boolean',
            'is_active' => 'boolean',
        ]);

        // Gérer la valeur selon le type
        $value = $this->processSettingValue($request, $validated);

        // Valider le JSON si nécessaire
        if ($validated['type'] === 'json' && $value) {
            json_decode($value);
            if (json_last_error() !== JSON_ERROR_NONE) {
                return back()->withErrors(['value_json' => 'La valeur doit être un JSON valide.']);
            }
        }

        $setting = Setting::create([
            'key' => $validated['key'],
            'label' => $validated['label'],
            'type' => $validated['type'],
            'value' => $value,
            'description' => $validated['description'],
            'group' => $validated['group'],
            'options' => $validated['options'],
            'order' => $validated['order'] ?? 0,
            'is_required' => $validated['is_required'] ?? false,
            'is_active' => $validated['is_active'] ?? true,
        ]);

        return redirect()->route('admin.settings.index')
            ->with('success', 'Paramètre créé avec succès.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Setting $setting)
    {
        return view('admin.settings.show', compact('setting'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Setting $setting)
    {
        $groups = ['general', 'contact', 'social', 'stats', 'seo', 'appearance'];
        $types = ['text', 'textarea', 'number', 'url', 'email', 'image', 'boolean', 'json'];
        
        return view('admin.settings.edit', compact('setting', 'groups', 'types'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Setting $setting)
    {
        $validated = $request->validate([
            'key' => 'required|string|max:255|unique:settings,key,' . $setting->id,
            'label' => 'required|string|max:255',
            'type' => 'required|string|in:text,textarea,number,email,url,tel,select,checkbox,file,image,color,date,time,json',
            'value' => 'nullable|string',
            'value_textarea' => 'nullable|string',
            'value_json' => 'nullable|string',
            'value_checkbox' => 'nullable|boolean',
            'file' => 'nullable|file|max:10240',
            'description' => 'nullable|string',
            'group' => 'required|string',
            'options' => 'nullable|string',
            'order' => 'nullable|integer',
            'is_required' => 'boolean',
            'is_active' => 'boolean',
        ]);

        // Gérer la valeur selon le type
        $value = $this->processSettingValueForUpdate($request, $validated, $setting);

        // Valider le JSON si nécessaire
        if ($validated['type'] === 'json' && $value) {
            json_decode($value);
            if (json_last_error() !== JSON_ERROR_NONE) {
                return back()->withErrors(['value_json' => 'La valeur doit être un JSON valide.']);
            }
        }

        $setting->update([
            'key' => $validated['key'],
            'label' => $validated['label'],
            'type' => $validated['type'],
            'value' => $value,
            'description' => $validated['description'],
            'group' => $validated['group'],
            'options' => $validated['options'],
            'order' => $validated['order'] ?? $setting->order,
            'is_required' => $validated['is_required'] ?? false,
            'is_active' => $validated['is_active'] ?? true,
        ]);

        return redirect()->route('admin.settings.index')
            ->with('success', 'Paramètre mis à jour avec succès.');
    }

    /**
     * Process setting value for update based on type
     */
    private function processSettingValueForUpdate(Request $request, array $validated, Setting $setting): ?string
    {
        switch ($validated['type']) {
            case 'textarea':
            case 'json':
                $fieldName = 'value_' . $validated['type'];
                return $request->input($fieldName);
                
            case 'checkbox':
                return $request->has('value_checkbox') ? '1' : '0';
                
            case 'file':
            case 'image':
                if ($request->hasFile('file')) {
                    // Supprimer l'ancien fichier
                    if ($setting->value) {
                        $this->deleteFile($setting->value);
                    }
                    
                    $file = $request->file('file');
                    $directory = $validated['type'] === 'image' ? 'settings/images' : 'settings/files';
                    return $this->uploadFile($file, $directory);
                }
                // Garder l'ancienne valeur si pas de nouveau fichier
                return $setting->value;
                
            default:
                return $validated['value'] ?? $setting->value;
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Setting $setting)
    {
        try {
            // Supprimer le fichier associé s'il s'agit d'un setting de type file ou image
            if (in_array($setting->type, ['file', 'image']) && $setting->value && \Storage::disk('public')->exists($setting->value)) {
                \Storage::disk('public')->delete($setting->value);
            }

            // Supprimer le paramètre
            $label = $setting->label;
            $setting->delete();

            return redirect()->route('admin.settings.index')
                ->with('success', "Le paramètre '{$label}' a été supprimé avec succès.");

        } catch (\Exception $e) {
            \Log::error('Erreur lors de la suppression du paramètre: ' . $e->getMessage());
            
            return redirect()->route('admin.settings.index')
                ->with('error', 'Une erreur est survenue lors de la suppression du paramètre.');
        }
    }

    /**
     * Process setting value based on type
     */
    private function processSettingValue(Request $request, array $validated): ?string
    {
        switch ($validated['type']) {
            case 'textarea':
            case 'json':
                $fieldName = 'value_' . $validated['type'];
                return $request->input($fieldName);
                
            case 'checkbox':
                return $request->has('value_checkbox') ? '1' : '0';
                
            case 'file':
            case 'image':
                if ($request->hasFile('file')) {
                    $file = $request->file('file');
                    $directory = $validated['type'] === 'image' ? 'settings/images' : 'settings/files';
                    return $this->uploadFile($file, $directory);
                }
                return null;
                
            default:
                return $validated['value'] ?? null;
        }
    }

    /**
     * Bulk update settings
     */
    public function bulkUpdate(Request $request)
    {
        $request->validate([
            'settings' => 'required|array',
            'settings.*.id' => 'required|exists:settings,id',
            'settings.*.value' => 'required|string',
        ]);

        foreach ($request->settings as $settingData) {
            $setting = Setting::find($settingData['id']);
            
            // Validation spécifique selon le type
            $value = $settingData['value'];
            if ($setting->type === 'boolean') {
                $value = $value ? '1' : '0';
            } elseif ($setting->type === 'json') {
                $json = json_decode($value);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    return response()->json([
                        'success' => false,
                        'message' => "Valeur JSON invalide pour {$setting->label}"
                    ], 422);
                }
            }
            
            $setting->update(['value' => $value]);
        }

        return response()->json([
            'success' => true,
            'message' => 'Paramètres mis à jour avec succès.'
        ]);
    }

    /**
     * Get public settings for frontend
     */
    public function getPublicSettings()
    {
        $settings = Setting::where('is_public', true)->get();
        
        $result = [];
        foreach ($settings as $setting) {
            $result[$setting->key] = $setting->parsed_value;
        }
        
        return response()->json($result);
    }

    /**
     * Clear cache
     */
    public function clearCache()
    {
        // Vider le cache des settings publics
        \Cache::forget('public_settings');
        
        // Vider le cache de tous les settings individuels (approximation)
        $settings = \App\Models\Setting::all();
        foreach ($settings as $setting) {
            \Cache::forget("setting_{$setting->key}");
        }
        
        return response()->json([
            'success' => true,
            'message' => 'Cache des paramètres vidé avec succès.'
        ]);
    }
}