<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class ProjectController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Project::query();

        // Filtres
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
            });
        }

        $projects = $query->orderBy('created_at', 'desc')->paginate(15);
        
        $statuses = ['planned', 'in_progress', 'completed', 'on_hold'];

        return view('admin.projects.index', compact('projects', 'statuses'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.projects.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'content' => 'nullable|string',
            'budget' => 'nullable|numeric|min:0',
            'progress_percentage' => 'required|integer|min:0|max:100',
            'status' => 'required|in:planned,in_progress,completed,on_hold',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'location' => 'nullable|string|max:255',
            'featured_image' => 'nullable|image|max:5120',
            'is_featured' => 'boolean',
        ]);

        // Génération du slug
        $validated['slug'] = Str::slug($validated['title']);
        
        // Assurer l'unicité du slug
        $originalSlug = $validated['slug'];
        $count = 1;
        while (Project::where('slug', $validated['slug'])->exists()) {
            $validated['slug'] = $originalSlug . '-' . $count;
            $count++;
        }

        // Gestion de l'image
        if ($request->hasFile('featured_image')) {
            $validated['featured_image'] = $request->file('featured_image')->store('projects', 'public');
        }

        $validated['is_featured'] = $request->has('is_featured') ? true : false;

        Project::create($validated);

        return redirect()->route('admin.projects.index')
            ->with('success', 'Projet créé avec succès.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Project $project)
    {
        return view('admin.projects.show', compact('project'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Project $project)
    {
        return view('admin.projects.edit', compact('project'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Project $project)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'content' => 'nullable|string',
            'budget' => 'nullable|numeric|min:0',
            'progress_percentage' => 'required|integer|min:0|max:100',
            'status' => 'required|in:planned,in_progress,completed,on_hold',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'location' => 'nullable|string|max:255',
            'featured_image' => 'nullable|image|max:5120',
            'is_featured' => 'boolean',
        ]);

        // Mise à jour du slug si le titre a changé
        if ($validated['title'] !== $project->title) {
            $validated['slug'] = Str::slug($validated['title']);
            
            // Assurer l'unicité du slug
            $originalSlug = $validated['slug'];
            $count = 1;
            while (Project::where('slug', $validated['slug'])->where('id', '!=', $project->id)->exists()) {
                $validated['slug'] = $originalSlug . '-' . $count;
                $count++;
            }
        }

        // Gestion de l'image
        if ($request->hasFile('featured_image')) {
            // Supprimer l'ancienne image
            if ($project->featured_image) {
                \Storage::disk('public')->delete($project->featured_image);
            }
            $validated['featured_image'] = $request->file('featured_image')->store('projects', 'public');
        }

        $validated['is_featured'] = $request->has('is_featured') ? true : false;

        $project->update($validated);

        return redirect()->route('admin.projects.index')
            ->with('success', 'Projet mis à jour avec succès.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Project $project)
    {
        try {
            // Supprimer l'image principale
            if ($project->image && \Storage::disk('public')->exists($project->image)) {
                \Storage::disk('public')->delete($project->image);
            }

            // Supprimer les images de galerie
            if ($project->gallery && is_array($project->gallery)) {
                foreach ($project->gallery as $image) {
                    if (\Storage::disk('public')->exists($image)) {
                        \Storage::disk('public')->delete($image);
                    }
                }
            }

            // Supprimer le projet
            $title = $project->title;
            $project->delete();

            return redirect()->route('admin.projects.index')
                ->with('success', "Le projet '{$title}' a été supprimé avec succès.");

        } catch (\Exception $e) {
            \Log::error('Erreur lors de la suppression du projet: ' . $e->getMessage());
            
            return redirect()->route('admin.projects.index')
                ->with('error', 'Une erreur est survenue lors de la suppression du projet.');
        }
    }

    /**
     * Update project progress
     */
    public function updateProgress(Request $request, Project $project)
    {
        $request->validate([
            'progress' => 'required|integer|min:0|max:100'
        ]);

        $project->update(['progress' => $request->progress]);

        return response()->json([
            'success' => true,
            'progress' => $project->progress,
            'progress_label' => $project->progress_label
        ]);
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:delete,activate,pause,complete',
            'projects' => 'required|array',
            'projects.*' => 'exists:projects,id'
        ]);

        $projects = Project::whereIn('id', $request->projects);

        switch ($request->action) {
            case 'delete':
                $projects->each(function ($project) {
                    if ($project->image) {
                        \Storage::disk('public')->delete($project->image);
                    }
                    if ($project->gallery) {
                        foreach ($project->gallery as $image) {
                            \Storage::disk('public')->delete($image);
                        }
                    }
                });
                $projects->delete();
                $message = 'Projets supprimés avec succès.';
                break;
            case 'activate':
                $projects->update(['status' => 'in_progress']);
                $message = 'Projets activés avec succès.';
                break;
            case 'pause':
                $projects->update(['status' => 'on_hold']);
                $message = 'Projets mis en pause avec succès.';
                break;
            case 'complete':
                $projects->update(['status' => 'completed', 'progress' => 100]);
                $message = 'Projets marqués comme terminés avec succès.';
                break;
        }

        return redirect()->route('admin.projects.index')
            ->with('success', $message);
    }
}