<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Page;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PageController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Page::query();

        // Filtres
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('content', 'like', '%' . $request->search . '%');
            });
        }

        $pages = $query->orderBy('created_at', 'desc')->paginate(15);
        
        $statuses = ['draft', 'published', 'archived'];

        return view('admin.pages.index', compact('pages', 'statuses'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.pages.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'excerpt' => 'nullable|string|max:500',
            'status' => 'required|in:draft,published,archived',
            'template' => 'nullable|string|max:100',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'meta_keywords' => 'nullable|string|max:500',
            'featured_image' => 'nullable|image|max:2048',
            'order' => 'nullable|integer|min:0',
            'show_in_menu' => 'boolean',
            'parent_id' => 'nullable|exists:pages,id',
        ]);

        // Génération du slug
        $validated['slug'] = Str::slug($validated['title']);
        
        // Assurer l'unicité du slug
        $originalSlug = $validated['slug'];
        $count = 1;
        while (Page::where('slug', $validated['slug'])->exists()) {
            $validated['slug'] = $originalSlug . '-' . $count;
            $count++;
        }

        // Gestion de l'image mise en avant
        if ($request->hasFile('featured_image')) {
            $validated['featured_image'] = $request->file('featured_image')->store('pages', 'public');
        }

        // Ordre automatique si non spécifié
        if (!isset($validated['order'])) {
            $validated['order'] = Page::max('order') + 1 ?? 0;
        }

        Page::create($validated);

        return redirect()->route('admin.pages.index')
            ->with('success', 'Page créée avec succès.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Page $page)
    {
        return view('admin.pages.show', compact('page'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Page $page)
    {
        $pages = Page::where('id', '!=', $page->id)->get();
        return view('admin.pages.edit', compact('page', 'pages'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Page $page)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'content' => 'required|string',
            'excerpt' => 'nullable|string|max:500',
            'status' => 'required|in:draft,published,archived',
            'template' => 'nullable|string|max:100',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'meta_keywords' => 'nullable|string|max:500',
            'featured_image' => 'nullable|image|max:2048',
            'order' => 'nullable|integer|min:0',
            'show_in_menu' => 'boolean',
            'parent_id' => 'nullable|exists:pages,id',
        ]);

        // Mise à jour du slug si le titre a changé
        if ($validated['title'] !== $page->title) {
            $validated['slug'] = Str::slug($validated['title']);
            
            // Assurer l'unicité du slug
            $originalSlug = $validated['slug'];
            $count = 1;
            while (Page::where('slug', $validated['slug'])->where('id', '!=', $page->id)->exists()) {
                $validated['slug'] = $originalSlug . '-' . $count;
                $count++;
            }
        }

        // Gestion de l'image mise en avant
        if ($request->hasFile('featured_image')) {
            // Supprimer l'ancienne image
            if ($page->featured_image) {
                \Storage::disk('public')->delete($page->featured_image);
            }
            $validated['featured_image'] = $request->file('featured_image')->store('pages', 'public');
        }

        // Vérifier que la page ne devient pas son propre parent
        if (isset($validated['parent_id']) && $validated['parent_id'] == $page->id) {
            return back()->withErrors(['parent_id' => 'Une page ne peut pas être son propre parent.']);
        }

        $page->update($validated);

        return redirect()->route('admin.pages.index')
            ->with('success', 'Page mise à jour avec succès.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Page $page)
    {
        try {
            // Vérifier s'il y a des pages enfants
            if ($page->children()->count() > 0) {
                return back()->withErrors(['error' => 'Impossible de supprimer une page qui a des sous-pages. Supprimez d\'abord les sous-pages.']);
            }

            // Supprimer l'image mise en avant
            if ($page->featured_image && \Storage::disk('public')->exists($page->featured_image)) {
                \Storage::disk('public')->delete($page->featured_image);
            }

            // Supprimer la page
            $title = $page->title;
            $page->delete();

            return redirect()->route('admin.pages.index')
                ->with('success', "La page '{$title}' a été supprimée avec succès.");

        } catch (\Exception $e) {
            \Log::error('Erreur lors de la suppression de la page: ' . $e->getMessage());
            
            return redirect()->route('admin.pages.index')
                ->with('error', 'Une erreur est survenue lors de la suppression de la page.');
        }
    }

    /**
     * Reorder pages
     */
    public function reorder(Request $request)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|exists:pages,id',
            'items.*.order' => 'required|integer|min:0',
        ]);

        foreach ($request->items as $item) {
            Page::where('id', $item['id'])->update(['order' => $item['order']]);
        }

        return response()->json(['success' => true]);
    }

    /**
     * Toggle menu visibility
     */
    public function toggleMenu(Page $page)
    {
        $page->update(['show_in_menu' => !$page->show_in_menu]);

        return response()->json([
            'success' => true,
            'show_in_menu' => $page->show_in_menu
        ]);
    }

    /**
     * Duplicate page
     */
    public function duplicate(Page $page)
    {
        $newPage = $page->replicate();
        $newPage->title = $page->title . ' (Copie)';
        $newPage->slug = Str::slug($newPage->title);
        
        // Assurer l'unicité du slug
        $originalSlug = $newPage->slug;
        $count = 1;
        while (Page::where('slug', $newPage->slug)->exists()) {
            $newPage->slug = $originalSlug . '-' . $count;
            $count++;
        }
        
        $newPage->status = 'draft';
        $newPage->save();

        return redirect()->route('admin.pages.edit', $newPage)
            ->with('success', 'Page dupliquée avec succès.');
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:delete,publish,draft,archive',
            'pages' => 'required|array',
            'pages.*' => 'exists:pages,id'
        ]);

        $pages = Page::whereIn('id', $request->pages);

        switch ($request->action) {
            case 'delete':
                $pages->each(function ($page) {
                    if ($page->children()->count() > 0) {
                        return; // Skip pages with children
                    }
                    if ($page->featured_image) {
                        \Storage::disk('public')->delete($page->featured_image);
                    }
                });
                $pages->whereDoesntHave('children')->delete();
                $message = 'Pages supprimées avec succès (les pages avec sous-pages ont été ignorées).';
                break;
            case 'publish':
                $pages->update(['status' => 'published']);
                $message = 'Pages publiées avec succès.';
                break;
            case 'draft':
                $pages->update(['status' => 'draft']);
                $message = 'Pages mises en brouillon avec succès.';
                break;
            case 'archive':
                $pages->update(['status' => 'archived']);
                $message = 'Pages archivées avec succès.';
                break;
        }

        return redirect()->route('admin.pages.index')
            ->with('success', $message);
    }
}