<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Faq;
use Illuminate\Http\Request;

class FaqController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Faq::query();

        // Filtres
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('question', 'like', '%' . $request->search . '%')
                  ->orWhere('answer', 'like', '%' . $request->search . '%');
            });
        }

        $faqs = $query->orderBy('order')->orderBy('created_at', 'desc')->paginate(15);
        
        $categories = Faq::select('category')->distinct()->pluck('category');

        return view('admin.faqs.index', compact('faqs', 'categories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = ['général', 'services', 'projets', 'partenariat', 'bénévolat'];
        return view('admin.faqs.create', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'question' => 'required|string|max:500',
            'answer' => 'required|string',
            'category' => 'required|string|max:100',
            'order' => 'nullable|integer|min:0',
            'is_featured' => 'boolean',
        ]);

        // Ordre automatique si non spécifié
        if (!isset($validated['order'])) {
            $validated['order'] = Faq::max('order') + 1 ?? 0;
        }

        Faq::create($validated);

        return redirect()->route('admin.faqs.index')
            ->with('success', 'FAQ créée avec succès.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Faq $faq)
    {
        return view('admin.faqs.show', compact('faq'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Faq $faq)
    {
        $categories = ['général', 'services', 'projets', 'partenariat', 'bénévolat'];
        return view('admin.faqs.edit', compact('faq', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Faq $faq)
    {
        $validated = $request->validate([
            'question' => 'required|string|max:500',
            'answer' => 'required|string',
            'category' => 'required|string|max:100',
            'order' => 'nullable|integer|min:0',
            'is_featured' => 'boolean',
        ]);

        $faq->update($validated);

        return redirect()->route('admin.faqs.index')
            ->with('success', 'FAQ mise à jour avec succès.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Faq $faq)
    {
        try {
            // Supprimer la FAQ
            $question = $faq->question;
            $faq->delete();

            return redirect()->route('admin.faqs.index')
                ->with('success', "La FAQ '{$question}' a été supprimée avec succès.");

        } catch (\Exception $e) {
            \Log::error('Erreur lors de la suppression de la FAQ: ' . $e->getMessage());
            
            return redirect()->route('admin.faqs.index')
                ->with('error', 'Une erreur est survenue lors de la suppression de la FAQ.');
        }
    }

    /**
     * Reorder FAQs
     */
    public function reorder(Request $request)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|exists:faqs,id',
            'items.*.order' => 'required|integer|min:0',
        ]);

        foreach ($request->items as $item) {
            Faq::where('id', $item['id'])->update(['order' => $item['order']]);
        }

        return response()->json(['success' => true]);
    }

    /**
     * Toggle featured status
     */
    public function toggleFeatured(Faq $faq)
    {
        $faq->update(['is_featured' => !$faq->is_featured]);

        return response()->json([
            'success' => true,
            'is_featured' => $faq->is_featured
        ]);
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:delete,feature,unfeature',
            'faqs' => 'required|array',
            'faqs.*' => 'exists:faqs,id'
        ]);

        $faqs = Faq::whereIn('id', $request->faqs);

        switch ($request->action) {
            case 'delete':
                $faqs->delete();
                $message = 'FAQs supprimées avec succès.';
                break;
            case 'feature':
                $faqs->update(['is_featured' => true]);
                $message = 'FAQs mises en avant avec succès.';
                break;
            case 'unfeature':
                $faqs->update(['is_featured' => false]);
                $message = 'FAQs retirées de la mise en avant avec succès.';
                break;
        }

        return redirect()->route('admin.faqs.index')
            ->with('success', $message);
    }
}