<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Document;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class DocumentController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Document::query();

        // Filtres
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('title', 'like', '%' . $request->search . '%')
                  ->orWhere('description', 'like', '%' . $request->search . '%');
            });
        }

        $documents = $query->orderBy('created_at', 'desc')->paginate(15);
        
        $categories = Document::select('category')->distinct()->pluck('category');

        return view('admin.documents.index', compact('documents', 'categories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.documents.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'required|in:reglement,statuts,rapport,formulaire,guide,procedure,autres',
            'file' => 'required|file|mimes:pdf,doc,docx,xls,xlsx,ppt,pptx,jpg,jpeg,png,gif|max:10240',
            'version' => 'nullable|string|max:50',
            'is_public' => 'nullable|boolean',
            'is_featured' => 'nullable|boolean',
        ]);

        // Gestion du fichier
        if ($request->hasFile('file')) {
            $file = $request->file('file');
            $filename = time() . '_' . Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $file->getClientOriginalExtension();
            $validated['file_path'] = $file->storeAs('documents', $filename, 'public');
            $validated['file_name'] = $file->getClientOriginalName();
            $validated['file_size'] = $file->getSize();
            $validated['file_type'] = strtolower($file->getClientOriginalExtension());
        } else {
            // Si pas de fichier, on ne peut pas créer de document
            return redirect()->back()->withErrors(['file' => 'Un fichier est requis.'])->withInput();
        }

        // Initialiser les champs manquants
        $validated['download_count'] = 0;
        $validated['is_public'] = $request->has('is_public') ? true : false;
        $validated['is_featured'] = $request->has('is_featured') ? true : false;
        $validated['icon_color'] = 'blue'; // Default color

        // Mapping des catégories vers celles de la DB
        $categoryMapping = [
            'reglement' => 'policy',
            'statuts' => 'policy',
            'rapport' => 'report',
            'guide' => 'guide', 
            'formulaire' => 'form',
            'procedure' => 'policy',
            'autres' => 'other'
        ];
        $validated['category'] = $categoryMapping[$validated['category']] ?? 'other';

        // Supprimer le champ 'file' car il n'existe pas dans la DB
        unset($validated['file']);

        Document::create($validated);

        return redirect()->route('admin.documents.index')
            ->with('success', 'Document créé avec succès.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Document $document)
    {
        return view('admin.documents.show', compact('document'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Document $document)
    {
        return view('admin.documents.edit', compact('document'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Document $document)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'category' => 'required|in:reglement,statuts,rapport,formulaire,guide,procedure,autres',
            'file' => 'nullable|file|mimes:pdf,doc,docx,xls,xlsx,ppt,pptx,jpg,jpeg,png,gif|max:10240',
            'version' => 'nullable|string|max:50',
            'is_public' => 'nullable|boolean',
            'is_featured' => 'nullable|boolean',
        ]);

        // Gestion du fichier
        if ($request->hasFile('file')) {
            // Supprimer l'ancien fichier
            if ($document->file_path) {
                \Storage::disk('public')->delete($document->file_path);
            }

            $file = $request->file('file');
            $filename = time() . '_' . Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $file->getClientOriginalExtension();
            $validated['file_path'] = $file->storeAs('documents', $filename, 'public');
            $validated['file_name'] = $file->getClientOriginalName();
            $validated['file_size'] = $file->getSize();
            $validated['file_type'] = strtolower($file->getClientOriginalExtension());
        }

        // Gestion des champs booléens
        $validated['is_public'] = $request->has('is_public') ? true : false;
        $validated['is_featured'] = $request->has('is_featured') ? true : false;

        // Mapping des catégories vers celles de la DB
        if (isset($validated['category'])) {
            $categoryMapping = [
                'reglement' => 'policy',
                'statuts' => 'policy',
                'rapport' => 'report',
                'guide' => 'guide', 
                'formulaire' => 'form',
                'procedure' => 'policy',
                'autres' => 'other'
            ];
            $validated['category'] = $categoryMapping[$validated['category']] ?? 'other';
        }

        // Supprimer le champ 'file' car il n'existe pas dans la DB
        unset($validated['file']);

        $document->update($validated);

        return redirect()->route('admin.documents.index')
            ->with('success', 'Document mis à jour avec succès.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Document $document)
    {
        try {
            // Supprimer le fichier
            if ($document->file_path && \Storage::disk('public')->exists($document->file_path)) {
                \Storage::disk('public')->delete($document->file_path);
            }

            // Supprimer le document
            $title = $document->title;
            $document->delete();

            return redirect()->route('admin.documents.index')
                ->with('success', "Le document '{$title}' a été supprimé avec succès.");

        } catch (\Exception $e) {
            \Log::error('Erreur lors de la suppression du document: ' . $e->getMessage());
            
            return redirect()->route('admin.documents.index')
                ->with('error', 'Une erreur est survenue lors de la suppression du document.');
        }
    }

    /**
     * Download document
     */
    public function download(Document $document)
    {
        if (!$document->file_path || !\Storage::disk('public')->exists($document->file_path)) {
            abort(404, 'Fichier introuvable.');
        }

        // Incrémenter le compteur de téléchargements
        $document->increment('downloads');

        return \Storage::disk('public')->download($document->file_path, $document->file_name);
    }

    /**
     * Toggle public status
     */
    public function togglePublic(Document $document)
    {
        $document->update(['is_public' => !$document->is_public]);

        return response()->json([
            'success' => true,
            'is_public' => $document->is_public
        ]);
    }

    /**
     * Bulk upload
     */
    public function bulkUpload(Request $request)
    {
        $request->validate([
            'files' => 'required|array',
            'files.*' => 'file|mimes:pdf,doc,docx,xls,xlsx,ppt,pptx|max:10240',
            'category' => 'required|string|max:100',
        ]);

        $uploaded = 0;

        foreach ($request->file('files') as $file) {
            $filename = time() . '_' . Str::slug(pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $file->getClientOriginalExtension();
            $filePath = $file->storeAs('documents', $filename, 'public');
            
            Document::create([
                'title' => pathinfo($file->getClientOriginalName(), PATHINFO_FILENAME),
                'category' => $request->category,
                'file_path' => $filePath,
                'file_name' => $file->getClientOriginalName(),
                'file_size' => $file->getSize(),
                'mime_type' => $file->getMimeType(),
                'is_public' => true,
            ]);
            
            $uploaded++;
        }

        return response()->json([
            'success' => true,
            'message' => "$uploaded documents téléchargés avec succès."
        ]);
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:delete,public,private',
            'documents' => 'required|array',
            'documents.*' => 'exists:documents,id'
        ]);

        $documents = Document::whereIn('id', $request->documents);

        switch ($request->action) {
            case 'delete':
                $documents->each(function ($document) {
                    if ($document->file_path) {
                        \Storage::disk('public')->delete($document->file_path);
                    }
                });
                $documents->delete();
                $message = 'Documents supprimés avec succès.';
                break;
            case 'public':
                $documents->update(['is_public' => true]);
                $message = 'Documents rendus publics avec succès.';
                break;
            case 'private':
                $documents->update(['is_public' => false]);
                $message = 'Documents rendus privés avec succès.';
                break;
        }

        return redirect()->route('admin.documents.index')
            ->with('success', $message);
    }
}